/*------------------------------------------------------------------------------*
 * File Name: MatrixData.h														*
 * Creation: TD 4-16-03															*
 * Purpose: Origin C header file for Origin basic Data matrix types				*
 * Copyright (c) OriginLab Corp.	2002 - 2007									*
 * All Rights Reserved															*
 * Modifications:
 *	Sim 07-05-2007 SHOULD_RETURN_NUMBER_AS_INT_BECAUSE_UNSIGNED_INT_0_MINUS_1_IS_NOT_EQUAL_-1
 *	Kyle 12/29/2008 QA80-12874 v8.0991b ADD_ARGUMENT_TO_RETURN_INDICES_OF_REMOVED_ROWS
 *	Kyle 07/08/2010 ORG-388-P1 FreqFilter_IN_matrixbase							*
 *------------------------------------------------------------------------------*/
#ifndef _MATRIXDATA_H
#define _MATRIXDATA_H

#ifndef _STRING_H
#include <string.h>		// Most likely will need strings
#endif // _STRING_H

#ifndef _OC_TYPES_H
#include <OC_types.h>	// Structures used in Origin internal functions
#endif

#ifndef _WKSHEET_H
#include <Wksheet.h>
#endif // _WKSHEET_H

#ifndef _VECTOR_H
#include <vector.h>
#endif


/////////////////////////////////////////////////////////////////////////
// matrixbase object
// Comments last updated by Frank on 03/08/2005
/**+
http://ocwiki.originlab.com/index.php?title=Category:matrixbase_(class)
*/
class matrixbase
{

protected:
	
	/**
		Objects of type matrixbase can not be constructed. This is an abstract base
		class used for polymorphic handling of all matrix and Matrix template types.
	*/
	matrixbase();	

public:
	
//Comments last updated by Joseph on 4/13/2007
 	/**+
_	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-SetSize
 	*/
	BOOL	SetSize(UINT wNumRows, UINT wNumCols, BOOL bKeepData = FALSE); // Set the size of any matrixbase derived object.

	// Comments last updated by Joseph on 04/13/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetNumRows
	*/
	///---Sim 07-05-2007 SHOULD_RETURN_NUMBER_AS_INT_BECAUSE_UNSIGNED_INT_0_MINUS_1_IS_NOT_EQUAL_-1
	//UINT	GetNumRows() const; 
	int		GetNumRows() const; 
	///---END SHOULD_RETURN_NUMBER_AS_INT_BECAUSE_UNSIGNED_INT_0_MINUS_1_IS_NOT_EQUAL_-1
	
	// Comments last updated by HF on 04/20/2005
 	/**+
_	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetNumCols
 	*/
	///---Sim 07-05-2007 SHOULD_RETURN_NUMBER_AS_INT_BECAUSE_UNSIGNED_INT_0_MINUS_1_IS_NOT_EQUAL_-1
	//UINT	GetNumCols() const; 
	int		GetNumCols() const; 
	///---END SHOULD_RETURN_NUMBER_AS_INT_BECAUSE_UNSIGNED_INT_0_MINUS_1_IS_NOT_EQUAL_-1

	// Comments last updated by Joseph on 04/13/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Padding
	*/
 	BOOL Padding( matrixbase &mPadding, int nPadSize, int nPaddingOption = MFILTER_ZEROPADDINGWINDOW ); // Pad a matrix with selected way

	// Comments last updated by Joseph on 04/13/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetInternalDataType
	*/
	int 	GetInternalDataType(); // Get the internal or underlying base data type of a matrixbase derived object.

	// Comments last updated by Joseph on 04/13/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-MedianFilter
	*/
	BOOL	MedianFilter(int nWindowSize, int nPaddingOption = MFILTER_ZEROPADDINGWINDOW); // Median filter.

	// Comments last updated by Frank on 03/08/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-ApplyFilter
	*/
	BOOL	ApplyFilter(matrix &mfilter, int nPaddingOption = MFILTER_ZEROPADDINGWINDOW, BOOL bNormalize = TRUE); // Apply filter.

	// Comments last updated by Joseph on 04/13/2007
 	/**+
_	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-FlipHorizontal
 	*/
	BOOL	FlipHorizontal(); // Horizontally flips this matrixbase derived object.
	
	// Comments last updated by HF on 04/26/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-FlipVertical
	*/
	BOOL	FlipVertical(); // Vertically flips this matrixbase derived object.

	// Comments last updated by Frank on 04/25/2005
 	/**+
_	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Rotate
 	*/
	BOOL	Rotate(int nAngleDegree = 90 ); // Rotates the matrix counter-clockwize by the specified angle.

	// Comments last updated by Joseph on 04/13/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-ReplaceLessThan
	*/
	BOOL	ReplaceLessThan(double dThresholdVal, double dReplaceVal); // Replace all cell values less than dThresholdVal with dReplaceVal.

	// Comments last updated by Joseph on 04/13/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-ReplaceGreaterThan
	*/
	BOOL	ReplaceGreaterThan(double dThresholdVal, double dReplaceVal); // Replace all cell values greater than dThresholdVal with dReplaceVal.

	// Comments last updated by HF on 05/09/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-SetColumn
	*/
	int		SetColumn(vectorbase &vb, UINT wIndexCol); // Set the values of a matrix column equal to the values of a vector.

	// Comments last updated by HF on 05/09/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-SetRow
	*/
	int		SetRow(vectorbase &vb, UINT wIndexRow); // Set the values of a matrix row equal to the values of a vector.

	// Comments last updated by HF on 05/09/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetColumn
	*/
	BOOL	GetColumn(vectorbase &vb, UINT wIndexCol); // Get the values of a matrix column and assign them to a vector.

	// Comments last updated by HF on 05/09/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetRow
	*/
	BOOL	GetRow(vectorbase &vb, UINT wIndexRow); // Get the values of a matrix row and assign them to a vector.

	// Comments last updated by HF on 05/10/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetAsVector
	*/
	BOOL	GetAsVector(vectorbase &vb, BOOL bByRow = TRUE); // Get the values of an entire matrix and assign them to a vector.

	// Comments last updated by HF on 05/10/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-SetByVector
	*/
	int		SetByVector(vectorbase &vb, BOOL bByRow = TRUE); // Set the values of a matrix equal to the values of a vector.

	// Comments last updated by HF on 05/10/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Reorder
	*/
	BOOL	Reorder(BOOL bRowOrder = TRUE); // Re-order a matrix from Column order to Row order or from Row order to Column order.

	// Comments last updated by HF on 05/10/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Transpose
	*/
	BOOL	Transpose(); // Transpose a matrix.
	
	// Comments last updated by HF on 05/10/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-DotMultiply
	*/
	BOOL	DotMultiply(matrixbase &mb); // Multiply two matrices element by element.

	// Comments last updated by HF on 05/10/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-DotDivide
	*/
	BOOL	DotDivide(matrixbase &mb); // Divide this matrix by another matrix.

	// Comments last updated by HF on 06/22/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-DotPower
	*/
	BOOL	DotPower(matrixbase &mb); // Raise this matrix to the power of another matrix.

	// Comments last updated by Frank on 03/08/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Sort
	*/
	void	Sort(BOOL bAscending = TRUE);

	// Comments last updated by HF on 6/07/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetMax
	*/
	double	GetMax(); // Returns the maximum cell value in the matrix.

	// Comments last updated by HF on 6/07/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetMin
	*/
	double	GetMin();

	// Comments last updated by Jake on 07/20/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetMedian
	*/
	double	GetMedian();

	// Comments last updated by Jake on 07/20/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetMean
	*/
	double	GetMean();

	// Comments last updated by Kyle on 12/29/2008
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-RemoveEmptyRows
	*/
	///Kyle 12/29/2008 QA80-12874 v8.0991b ADD_ARGUMENT_TO_RETURN_INDICES_OF_REMOVED_ROWS
	//int		RemoveEmptyRows(BOOL bOnlyRemoveEmptyRows = TRUE);
	int		RemoveEmptyRows(BOOL bOnlyRemoveEmptyRows = TRUE, vector<uint>& vnIndicesRemovedRows = NULL);
	///End ADD_ARGUMENT_TO_RETURN_INDICES_OF_REMOVED_ROWS

	// Comments last updated by HF on 05/11/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Shrink
	*/
	BOOL	Shrink( int nColShrinkFactor, int nRowShrinkFactor );

	// Comments last updated by HF on 07/19/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetSubMatrix
	*/
	BOOL	GetSubMatrix( matrixbase& mbTarget, int c1 = 0, int c2 = -1, int r1 = 0, int r2 = -1); // -1 means to last col/row

	// Comments last updated by HF on 05/11/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-SetSubMatrix
	*/
	BOOL	SetSubMatrix(matrixbase& mbSource, int c1 = 0, int r1 = 0);
	
	// Comments last updated by HF on 05/11/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-CopyFromWks
	*/
	BOOL	CopyFromWks(Worksheet & wksSource, int c1 = 0, int c2 = -1, int r1 = 0, int r2 = -1); // Copies a specified range from a worksheet to a matrix.

	
	
	// Comments last updated by Arvin on 02/12/2007
	// Comments last updated by HF on 05/12/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Replace
	*/
	///Arvin 02/12/07 REPLACE_SHOULD_SUPPORT_ABSOLUTE_ERROR	 by suggestion of Justin and larry
	//BOOL	Replace( double dThresholdVal, double dReplaceVal = 0, uint wBitwiseOptions = 0, int nTolerance = 4 );
	BOOL	Replace( double dThresholdVal, double dReplaceVal = 0, uint wBitwiseOptions = 0, double dAbsError = 0.01);
	///END 	REPLACE_SHOULD_SUPPORT_ABSOLUTE_ERROR
	// Comments last updated by HF on 7/5/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetReal
	*/
	BOOL	GetReal( matrix &mReal ); // Get the Real part of a Complex matrix.

	// Comments last updated by HF on 7/5/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetImaginary
	*/
	BOOL	GetImaginary( matrix &mImag ); // Get the Imaginary part of a Complex matrix. 
	
	// Comments last updated by HF on 7/5/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetPhase
	*/
	BOOL	GetPhase( matrix& mPhase ); // Get the phase angle matrix, in radians, of the Complex matrix.

	// Comments last updated by HF on 7/5/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetAmplitude
	*/
	BOOL	GetAmplitude( matrix& mAmplitude ); // Get the amplitude(modulus) of the Complex matrix.

	// Leo 06/15/05 QA7715 QA6836 updated
	// Comments last updated by HF on 05/12/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Inverse
	*/
	int		Inverse(); // Replace this matrix with the Inverse of this matrix.
	
	///Arvin 12/22/06 ADD_BASIC_MORPHOLOGICAL_OPERATIONS_IN_MATRIXBASE
	//Comments last updated by Jake on 07/20/2007
	/**#
		Dilates grayscale and binary image, this function only be valid in double type matrix 
	Keywords:
		Morphological operation, dilate, erode, open, close
	Examples1:
		void matrixbase_ImDilate_ex1()
		{
			matrix matData = {{0,0,0,0,0},
							  {0,1,2,3,0},
							  {0,2,3,4,0},
							  {0,3,4,5,0},
							  {0,0,0,0,0}};
				
			matrix matHood =  {{0,1,0},{1,2,1},{0,1,0}};
			matrix matHeight;
			matHeight = matHood;
			StrelResult se;
			se.nRows = 3;
			se.nCols = 3;
			se.pNHoodMat = matHood;
			se.pHeightMat = matHeight;
			int nRet = matData.ImDilate(&se);
			if(nRet != OE_NOERROR)
				printf("  Error: ImDilate failed. Error Code=%d\n", nRet);
		}
		
		void matrixbase_ImDilate_ex2()
		{
			matrix matData = {{0,0,0,0,0},
							  {0,1,2,3,0},
							  {0,2,3,4,0},
							  {0,3,4,5,0},
							  {0,0,0,0,0}};
			//see the definition of StrelParams in ocmath.h
			matrix matHood =  {{0,1,0},{1,2,1},{0,1,0}};
			matrix matHeight;
			matHeight = matHood;
			StrelParams sp;
			sp.nRows = 3;
			sp.nCols = 3;
			sp.pNHood = matHood;
			sp.pHeight = matHeight;
			
			matrix mNHood, mHeight;
			mNHood.SetSize(3,3);
			mHeight.SetSize(3,3);
			StrelResult se;
			se.nRows = 3;
			se.nCols = 3;
			se.pNHoodMat = mNHood;
			se.pHeightMat = mHeight;
			int nRet = ocmath_make_strel(&se, &sp, ARBITRARY_SHAPE);
			if(nRet != OE_NOERROR)
			{
				printf("  Error: Make strel failed. Error Code=%d\n", nRet);
				return;
			}
			
			nRet = matData.ImDilate(&se);
			if(nRet != OE_NOERROR)
				printf("  Error: ImDilate failed. Error Code=%d\n", nRet);
		}
	Parameters:
		se = [input] pointer to the object of structure StrelResult. StrelResult defines a morphological structuring element. 
			         It has 4 elements:
					 pNHoodMat: pointer to neighborhood matrix. The neighborhood matrix specifies the neighborhood.
					 pHeightMat: pointer to height matrix. The height matrix must be real and finite valued and same size as the neighborhood matrix.
					 			 When it is NULL or all cells of it equals 0, se is flat structuring element, or else nonflat.  
					 nRows: the row number of matrix;
					 nCols: the column number of matrix;
	Return:
		Returns 0 on success and a non-zero error code on failure.
	SeeAlso:
		matrixbase::ImErode, matrixbase::ImOpen, matrixbase::ImClose
	*/
	int 	ImDilate(StrelResult* se);
	/// Last updated by Sim, 08-01-2008 need # for each method
	/**#
	*/
	int 	ImErode(StrelResult* se);
	/// Last updated by Sim, 08-01-2008 need # for each method
	/**#
	*/
	int 	ImOpen(StrelResult* se);
	/// Last updated by Sim, 08-01-2008 need # for each method
	/**#
	*/
	int 	ImClose(StrelResult* se);
	///end ADD_BASIC_MORPHOLOGICAL_OPERATIONS_IN_MATRIXBASE
	
	// Comments last updated by HF on 6/23/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Conjugate
	*/
	int		Conjugate(); // Replace this matrix with the Conjugate of this matrix.

	// Comments last updated by HF on 7/21/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-MakeComplex
	*/
	int		MakeComplex(matrixbase & mbReal, matrixbase & mbImag); // Make a Complex matrix from two Real matrices.

	// Comments last updated by Frank on 03/08/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-SumColumns
	*/
	int		SumColumns(vectorbase & vbSum); // Sum the columns of this matrix placing the result in a vector.
   
	// Comments last updated by HF on 05/27/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Concatenate
	*/
	int		Concatenate(int nDim, matrixbase & mSource, matrixbase & mResult ); // Concatenate a matrix to this matrix placing the results in a third matrix. 

	// Comments last updated by HF on 7/21/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-FFTShift
	*/
	int		FFTShift(matrixbase & mbShifted,  int nDim = -1); // Shift this matrix placing the result in the matrix passed as an argument.

	// Comments last updated by HF on 7/21/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-IFFTShift
	*/
	int		IFFTShift(matrixbase & mbShifted,  int nDim = -1); // Inverse FFTShift.

    /// Comments last updated by Derek on 08/06/2010
	///Kyle 07/08/2010 ORG-388-P1 FreqFilter_IN_matrixbase
	/**
		Generation Function to Create Filter
	Keywords:
		Matrix, Filter, Ideal, Gaussian, Butterworth, Blackman
	Parameters:
		nRows=[input]number of rows
		nCols=[input]number of columns
		dwType=[input] Options including 4 fields: window type; filter type; whether to cut the values outside the circle; cutoff is specified in
		               fraction or Fourier pixel. The bit fields are defined in tagFILTER2DWINDOW.		
		dCutOff=[input]the cutoff value, specified in fraction or Fourier pixel. If in fraction, the value is within range (0, 0.5]; If in Fourier
		               pixel, the value is an arbitrary positive number of double type
		nOrder=[input]the filter order for Butterworth
		dCutOff2=[input]the second cutoff value for FILTER2D_BAND_PASS or FILTER2D_BAND_BLOCK, should make sure dCutOff2 >= dCutOff
	Examples1:
		int matrixbase_FreqFilter_ex1()
		{
			matrix		mat;
			int nRet = mat.FreqFilter(32, 32, FILTER2D_GAUSSIAN | FILTER2D_HIGH_PASS | FILTER2D_CUTOFF_IN_FRACTION);
			ASSERT( FILTER2DERROR_NO_ERROR == nRet );
			return nRet;
		}
	Return:
		Returns FILTER2DERROR_NO_ERROR on successful and a non-zero error code on failure, see FILTER2DERROR.
	*/
	int		FreqFilter(uint nRows, uint nCols, DWORD dwType = FILTER2D_BUTTERWORTH | FILTER2D_LOW_PASS |FILTER2D_CUTOFF_IN_FRACTION , double dCutOff = 0.5, int nOrder = 1, double dCutOff2 = 0.5);
	///End FreqFilter_IN_matrixbase


	// Comments last updated by HF on 06/02/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Floor
	*/
	int		Floor(matrixbase & mbFloor); // Round each cell of this matrix to the nearest integer less than the current cell value (toward minus infinity).

	// Comments last updated by HF on 6/30/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Fix
	*/
	int		Fix(matrixbase & mbFix); // Round each cell of this matrix to the nearest integer (toward zero).

	// Comments last updated by HF on 6/30/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Round
	*/
	int		Round(matrixbase & mbRound); // und each cell of this matrix to the nearest integer (absolute nearest).
	
	// Comments last updated by HF on 6/30/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Ceil
	*/
	int		Ceil(matrixbase & mbCeil); // Round each cell of this matrix to the nearest integer greater than the current cell value (toward positive infinity).

	// Comments last updated by HF on 6/28/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-MakeIdentity
	*/
	int		MakeIdentity(int mNumRows, int nNumCols = -1);

	// Comments last updated by HF on 07/19/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-CastToDouble
	*/
	int		CastToDouble(matrix & mDouble); // Copy this matrix casting the underlying base type to double.

	// Comments last updated by HF on 7/19/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-CastToInteger
	*/
	int		CastToInteger(matrix<int> & mInteger); // Round this matrix placing the results in a matrix having an underlying base type of int.

	// Comments last updated by HF on 6/09/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetLowerTriangular
	*/
	int		GetLowerTriangular(matrixbase & mbResult, int nNumDiagonal = 0); // Get a lower triangular matrix from this matrix.

	// Comments last updated by HF on 6/10/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetUpperTriangular
	*/
	int		GetUpperTriangular(matrixbase & mbResult, int nNthDiagonal = 0); // Get an upper triangular matrix from this matrix. 

	// Comments last updated by HF on 6/13/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-GetDiagonal
	*/
	int		GetDiagonal(vectorbase & vbDiagonal, int nNthDiagonal = 0); // Get the diagonal or N-th diagonal of this matrix.

	// Comments last updated by HF on 6/13/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-SetDiagonal
	*/
	int		SetDiagonal(vectorbase & vbDiagonal, int nNthDiagonal = 0); // Set the diagonal or N-th diagonal of this matrix.

	// Comments last updated by HF on 05/18/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Cross
	*/
	int		Cross(matrixbase & mbSource, int nDim = 1);

	// Comments last updated by Frank on 7/6/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-AllNonZero
	*/
	int		AllNonZero(vector<BOOL>  & vAllNonZero, int nDim  = 1, double dTol = DEFAULT_TOLERANCE); // Indicates whether or not an entire row or column of this matrix is comprised of all non-zero elements.

	// Comments last updated by Joseph on 4/13/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-AnyNonZero
	*/
	int		AnyNonZero(vector<BOOL> & vAnyNonZero, int nDim = 1, double dTol = DEFAULT_TOLERANCE); // Indicates whether or not each row or column of this matrix contains any non-zero elements.

	// Comments last updated by HF on 05/18/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-CumulativeProduct
	*/
	int		CumulativeProduct(matrixbase & mbProduct, int nDim = 1); 

	// Comments last updated by HF on 6/23/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-CumulativeSum
	*/
	int		CumulativeSum(matrixbase & mbSum, int nDim = 1);

	// Comments last updated by HF on 7/19/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Difference
	*/
	int		Difference(matrixbase & mbDifference, int nOrder = 1, int nDim = 1); // Compute the N-th order row wise or column wise difference of the elements in this matrix.
	
	// Comments last updated by HF on 7/21/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Wrap
	*/
	BOOL Wrap(int nRowNum=0, int nColNum =0); 


#if  _OC_VER > 0x0703


#ifdef	ORIGIN_COM_SUPPORT
	// Comments last updated by Frank on 03/08/2005
	/**#
		Set matrix elements from variant encountered in LabView's data.
		This function is available only for OriginPro versions, or with a special COM enabled license
	Keywords:
		Matrix, Variant
	Parameters:
		v=[input] variant that is VT_ARRAY|VT_VARIANT type and subvariants are of VT_ARRAY|<numeric> type.
	Examples1:
		#include <variant.h>	
		#include <VariantTypes.h>
		int matrixbase_SetByArrayInVariant_ex1()
		{
			matrix		mat;        
			
			_VARIANT	var00, var01, var10, var11;
			var00 = 11, var01 = 12;
			var10 = 21, var11 = 22;
			
			_VARIANT	var0, var1;
			var0.CreateAsArray( VT_VARIANT, 2 );
			var0.SetOneVariantInArray( var00, 0 );
			var0.SetOneVariantInArray( var10, 1 );
			var1.CreateAsArray( VT_VARIANT, 2 );
			var1.SetOneVariantInArray( var01, 0 );
			var1.SetOneVariantInArray( var11, 1 );
		
			_VARIANT	var;
			var.CreateAsArray( VT_VARIANT, 2 );
			var.SetOneVariantInArray( var0, 0 );
			var.SetOneVariantInArray( var1, 1 );
			
			BOOL	bRet = mat.SetByArrayInVariant( var );
			ASSERT(bRet);
			return 0;
		}
	Return:
		Returns TRUE on successful and FALSE on failure.			
	*/
	BOOL SetByArrayInVariant(_VARIANT v);

	// Comments last updated by Frank on 03/08/2005
	/**#
		This function is available only for OriginPro versions, or with a special COM enabled license
	Keywords:
		Matrix, 2DArray; 2DArray; Get data by COM
	Examples1:
		#include <variant.h>	
		#include <VariantTypes.h>
		int matrixbase_GetAs2DArray_ex1()
		{
			matrix mat = {{1,2,3,4,5}, {6,7,8,9,10}, {11,12,13,14,15}};
			
			_VARIANT	var;
			var = mat.GetAs2DArray();
			return 0;
		}
	Return:
		variant that is VT_ARRAY|VT_VARIANT type and subvariants are of VT_ARRAY|<numeric> type.			
	*/
	_VARIANT	GetAs2DArray();

#endif //	ORIGIN_COM_SUPPORT

	///Comments last updated by Joseph on 12/23/2006
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-PutRecordset
	*/
	BOOL	PutRecordset(Object &objrs, int nMaxNumFields = -1, int nMaxNumRecords = -1, int nBufferSize = 10);



	// Comments last updated by HF on 6/14/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-Find
	*/
	int Find(vector<uint> &vecR, vector<uint> &vecC, double dLower, double dUpper = _ONAN, int nPrecision = 8, int r1 = 0, int c1 = 0,int r2 = -1,int c2 = -1);


	// Comments last updated by HF on 7/19/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-CopyTo
	*/
	///BOOL	CopyTo(Worksheet& wksTarget, int r1 = 0, int c1 = 0, int r2 = -1, int c2 = -1);
	BOOL	CopyTo(Worksheet& wksTarget, int r1 = 0, int c1 = 0, int r2 = -1, int c2 = -1, int nr1Wks = 0, int nc1Wks = 0, BOOL bTransposed = TRUE, BOOL bSetColumnTypes = TRUE);

	
	/**#
			matrixbase_Copy a subset of this matrix specified by 0 based column and row indices to a specified matrix
		Keywords:
			Matrix, CopyTo
		Example1:
			int matrixbase_CopyTo_ex1()
			{
				matrix<int> m1;
				m1.SetSize( 10, 10 );
				for(int jj = 0; jj < 10; jj++)
					for(int ii = 0; ii < 10; ii++)
						m1[ii][jj] = ii*jj;
				
				matrix m2;

				m1.CopyTo( m2, 5, 5); 
				ASSERT( m2.GetNumCols() == 5 );
				ASSERT( m2.GetNumRows() == 5 );
				
				m1.CopyTo( m2, 5, 5, 6, 6); 
				ASSERT( m2.GetNumCols() == 2 );
				ASSERT( m2.GetNumRows() == 2 );	

				ASSERT(m2[0][0] == 25);
				ASSERT(m2[0][1] == 30);
				ASSERT(m2[1][0] == 30);
				ASSERT(m2[1][1] == 36);
				return 0;
			}
		Parameters: 
			matTarget = [output] matrix object to receive result 
			r1= [input] Begining row index, default is 0 (0 based offset) 
			c1= [input] Begining column index, default is 0 (0 based offset) 
			r2= [input] Ending row index, (inclusive) default -1 is GetNumRows -1 (0 based offset)
			c2= [input] Ending column index, (inclusive) default -1 is GetNumCols -1 (0 based offset) 
		Return:
			Returns TRUE on successful exit and FALSE on failure.
		*/
	BOOL	CopyTo(matrixbase& matTarget, int r1 = 0, int c1 = 0, int r2 = -1, int c2 = -1);

	/**#
			Copies a specified range from a worksheet to a matrix dynamically
			resizing the matrix as needed.
		Keywords:
			Matrix, CopyFrom
		Example1:
			int matrixbase_CopyFrom_ex1()
			{
				// Assumes Data1 worksheet and Matrix1 matrix exist in Origin
				Worksheet wks("Data1");
				Matrix mat1("Matrix1");
				BOOL bRet = TRUE;
				if(mat1 && wks)
					bRet = mat1.CopyFrom(wks); //Copies the whole wks to m1
				return 0;
			}
		Parameters: 
			wksSource=[input] Source worksheet from which data is copied
			r1= [input] Begining row index, default is 0 (0 based offset) 
			c1= [input] Begining column index, default is 0 (0 based offset) 
			r2= [input] Ending row index, (inclusive) default -1 is GetNumRows -1 (0 based offset)
			c2= [input] Ending column index, (inclusive) default -1 is GetNumCols -1 (0 based offset) 
		Return:
			Returns TRUE on successful exit and FALSE on failure.
		*/
	BOOL	CopyFrom(Worksheet & wksSource, int r1 = 0, int c1 = 0, int r2 = -1, int c2 = -1); 
	
	/**#
			Set this matrix by another one specified by 0 based column and row indices.
			The source and target matrices need to have the same data type, otherwise,
			this method will return false.
		Keywords:
			Matrix, CopyFrom
		Example1:
			int matrixbase_CopyFrom_ex1()
			{
				Matrix mat1("Matrix1");
				Matrix mat2("Matrix2");			
				mat2.CopyFrom(mat1, 10, 10);
				return 0;
			}
		Parameters: 
			mbSource =[input] matixbase oject containing the subset to set
			r1 = [input] Begining row index in the target matrix, default is 0 (0 based offset) 
			c1 = [input] Begining column index in the target matrix, default is 0 (0 based offset) 
		Return:
		 	Returns TRUE on successful exit and FALSE on failure.
	*/
	BOOL	CopyFrom(matrixbase& mbSource, int r1 = 0, int c1 = 0);



#endif //  _OC_VER > 0x0703
	// Comments last updated by HF on 7/21/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrixbase-ReorderRowWise
	*/
	BOOL ReorderRowWise(const vector<UINT>& vnIndices, int nC1 = 0, int nC2 = -1 );


};

/////////////////////////////////////////////////////////////////////////
// matrix (with lower case m) object
//Comments last updated by Jake on 07/20/2007
/**+
http://ocwiki.originlab.com/index.php?title=Category:matrix_(lowcase)_(class)
*/
class matrix : public matrixbase
{

public:
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrix_(lowcase)-matrix
	*/
	matrix(); // Default constructor for matrix class.

	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrix_(lowcase)-matrix
	*/
	matrix(matrixbase & mbOriginal); // Copy constructor for matrix class.

	///Comments last updated by Jake on 07/23/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrix_(lowcase)-matrix
	*/
	matrix(Dataset dsDataset); // Constructor for matrix class that constructs a matrix object from a Dataset object.
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrix_(lowcase)-matrix
	*/
	matrix(UINT nNumRows, UINT nNumCols); // Constructor for matrix class that constructs a matrix having a specific size.

#if  _OC_VER > 0x0703

	///Comments last updated by Jake on 07/23/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrix_(lowcase)-matrix
	*/
	matrix(MatrixObject &mo,  BOOL bWriteback = FALSE);	///Frank 9/12/05 CP said no need this function now, can use matrixbase::GetDataObject to replace.
	
	//Comments last updated by Joseph on 7/25/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:matrix_(lowcase)-GetSourceDim
	*/
	bool	GetSourceDim(int &nRows, int &nCols);
#endif //#if  _OC_VER > 0x0703

};

/////////////////////////////////////////////////////////////////////////
// Matrix (with upper case M) object
//Comments last updated by Joseph on 7/25/2007
/**+
http://ocwiki.originlab.com/index.php?title=Category:Matrix_(class)
*/
class Matrix : public matrix
{

public:

	//Comments last updated by Joseph on 7/25/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-Matrix
	*/
	Matrix(); // Default constructor for Matrix class.
	
	//Comments last updated by Joseph on 7/25/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-Matrix
	*/
	Matrix(LPCSTR lpcszMatrixName); // Constructor for Matrix class that attaches to an internal Origin matrix by name.
	
	//Comments last updated by Joseph on 7/25/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-Matrix
	*/
	Matrix(MatrixLayer& mlMatLayer, int nMatObj = -1); // Constructor for Matrix class that attaches to an internal Origin matrix by a MatrixLayer object.
 
	//Comments last updated by Joseph on 7/25/2007
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-Attach
	*/
	BOOL	Attach(LPCSTR lpcszMatrixName); // Attach matrix object to internal Origin matrix
	
	// Comments last updated by HF on 6/28/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-Attach
	*/
	BOOL	Attach(MatrixLayer& mlMatLayer, int nMatObj = -1); // Attach a Matrix object to an internal Origin matrix by a MatrixLayer object.

	// Comments last updated by HF on 6/30/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-Detach
	*/
	BOOL	Detach(); // Detach a Matrix object from an internal Origin matrix.
	
	// Comments last updated by HF on 6/07/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetXMin
	*/
	double	GetXMin() const; // Get the X coordinate associated with the leftmost column of the matrix.

	// Comments last updated by HF on 6/07/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetXMax
	*/
	double	GetXMax() const; // Get the X coordinate associated with the right most column of the matrix.

	// Comments last updated by HF on 6/07/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetYMin
	*/
	double	GetYMin() const; // Get the Y coordinate associated with the first row of the matrix.

	// Comments last updated by HF on 05/26/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetYMax
	*/
	double	GetYMax() const; // Get the Y coordinate associated with the last row of the matrix.

	// Comments last updated by HF on 6/07/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-SetXMin
	*/
	BOOL    SetXMin(double dXMin); // Set the X coordinate associated with the left most column of the matrix.

	// Comments last updated by HF on 6/07/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-SetYMin
	*/
	BOOL    SetYMin(double dYMin); // Set the Y coordinate associated with the first row of the matrix.

	// Comments last updated by HF on 6/07/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-SetXMax
	*/
	BOOL    SetXMax(double dXMax); // Set the X coordinate associated with the right most column of the matrix.

	// Comments last updated by HF on 05/26/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-SetYMax
	*/
	BOOL    SetYMax(double dYMax); // Set the Y coordinate associated with the last row of the matrix.

	// Comments last updated by HF on 06/27/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetXValue
	*/
	double	GetXValue(int iColumn); // Get the X coordinate associated with the specified column of the matrix.
	
	// Comments last updated by HF on 06/30/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetYValue
	*/
	double	GetYValue(int iRow); // Get the Y coordinate associated with the specified row of the matrix.

	// Comments last updated by HF on 05/26/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-SetCellValue
	*/
	BOOL	SetCellValue(double x, double y, double dValue); // Set the value of the cell nearest the specified X and Y coordinates.
	

	// Comments last updated by HF on 06/30/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetCellValue
	*/
	double	GetCellValue(double x, double y); // Get the value of the cell nearest the specified X and Y coordinates.


	// Comments last updated by HF on 6/6/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-IsValid
	*/
	BOOL	IsValid(); // Checks the validity of this Matrix.

	// Comments last updated by HF on 7/12/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-SetZRangeUpdate
	*/
	void SetZRangeUpdate(BOOL bSet = TRUE); // Enable or disable the automatic dynamic range (Z) update when Matrix data is changed.

	// Comments last updated by HF on 7/12/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-SetZRange
	*/
	BOOL SetZRange(double z1, double z2, BOOL bRound=FALSE); // Set the dynamic range of this Matrix (Z1, Z2) that is used in data to image conversion.

	// Comments last updated by HF on 7/12/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetZRange
	*/
	BOOL GetZRange(double& z1, double& z2); // Get the dynamic range (Z1, Z2) that is used in the data to image conversion.


	// Comments last updated by HF on 7/14/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-Update
	*/
	void	Update(BOOL bFromOrigin, int mode = REDRAW_REFRESH); // Update an Origin C Matrix object from an Origin matrix or vice-versa. 

	// Comments last updated by HF on 7/14/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetName
	*/
	string  GetName(); // Get the name of an internal Origin Matrix.

	// Comments last updated by HF on 7/18/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-ImageLinesProfile
	*/
	int		ImageLinesProfile(vector<fpoint>& vPathVertices, vector<fpoint>& vCalPoints, vectorbase & vIntensity,
			int iPointNumber = -1, int iMethod = INTERPOLATE_NEAREST); // Get intensity values over the course of a user defined path through this Matrix.

	// Comments last updated by HF on 7/19/2005
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:Matrix-GetInterpolatedValue
	*/
	double	  GetInterpolatedValue(double dX, double dY, int nMethod = INTERPOLATE_NEAREST, int* pnError = NULL); // Get an interpolated Z value from an internal Origin matrix.
};

#endif //_MATRIXDATA_H
